<?php
session_start();

// Configurações do site
define('SITE_NAME', 'VOD Generator');
define('SITE_URL', 'http://localhost/vod-generator');
define('ADMIN_EMAIL', 'admin@vodgenerator.com');

// Cores do tema
define('COLOR_METALIC', '#2c3e50');
define('COLOR_NEON', '#00ffff');

// TMDB API Key
define('TMDB_API_KEY', 'a688ebacd173967acc7359daa726e3d3');
define('TMDB_IMAGE_URL', 'https://image.tmdb.org/t/p/w500');
define('TMDB_IMAGE_ORIGINAL', 'https://image.tmdb.org/t/p/original');

// Diretórios
define('ROOT_PATH', __DIR__);
define('UPLOAD_DIR', ROOT_PATH . '/assets/images/');
define('BANNERS_DIR', UPLOAD_DIR . 'banners/');
define('CARDS_DIR', UPLOAD_DIR . 'cards/');
define('LOGOS_DIR', UPLOAD_DIR . 'logos/');
define('PROFILES_DIR', UPLOAD_DIR . 'profiles/');
define('FONTS_DIR', ROOT_PATH . '/assets/fonts/');
define('GAMES_CARDS_DIR', UPLOAD_DIR . 'games_cards/');
define('GAMES_IMAGES_DIR', UPLOAD_DIR . 'games_images/');

// Criar diretórios se não existirem
$dirs = [UPLOAD_DIR, BANNERS_DIR, CARDS_DIR, LOGOS_DIR, PROFILES_DIR, FONTS_DIR, GAMES_CARDS_DIR, GAMES_IMAGES_DIR];
foreach ($dirs as $dir) {
    if (!file_exists($dir)) {
        mkdir($dir, 0777, true);
    }
}

// Credenciais admin (usuário: admin, senha: admin123)
define('ADMIN_USER', 'admin');
define('ADMIN_PASS', password_hash('admin123', PASSWORD_DEFAULT));

// Configurações de upload
define('MAX_FILE_SIZE', 5242880); // 5MB
define('ALLOWED_EXTENSIONS', ['png', 'jpg', 'jpeg', 'gif']);

// Arquivos JSON para dados
define('MOVIES_FILE', ROOT_PATH . '/data/movies.json');
define('SERIES_FILE', ROOT_PATH . '/data/series.json');
define('THEMES_FILE', ROOT_PATH . '/data/themes.json');
define('THEMES_VOD_FILE', ROOT_PATH . '/data/themes_vod.json');
define('GAMES_THEMES_FILE', ROOT_PATH . '/data/games_themes.json');
define('GAMES_FILE', ROOT_PATH . '/data/games.json');
define('SETTINGS_FILE', ROOT_PATH . '/data/settings.json');
define('CARDS_FILE', ROOT_PATH . '/data/cards.json');

// Criar diretório data
if (!file_exists(ROOT_PATH . '/data')) {
    mkdir(ROOT_PATH . '/data', 0777, true);
}

// Criar arquivos JSON se não existirem
$json_files = [MOVIES_FILE, SERIES_FILE, THEMES_FILE, THEMES_VOD_FILE, GAMES_THEMES_FILE, GAMES_FILE, SETTINGS_FILE, CARDS_FILE];
foreach ($json_files as $file) {
    if (!file_exists($file)) {
        file_put_contents($file, json_encode([]));
    }
}

// Funções de sessão
function isLoggedIn() {
    return isset($_SESSION['admin_logged_in']) && $_SESSION['admin_logged_in'] === true;
}

function redirect($url) {
    header("Location: $url");
    exit();
}

// Funções para manipular JSON
function readJSON($file) {
    if (!file_exists($file)) {
        return [];
    }
    $content = file_get_contents($file);
    return !empty($content) ? json_decode($content, true) : [];
}

function writeJSON($file, $data) {
    return file_put_contents($file, json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
}

// Função para buscar dados do TMDB
function searchTMDB($query, $type = 'movie') {
    $url = "https://api.themoviedb.org/3/search/{$type}?api_key=".TMDB_API_KEY."&query=".urlencode($query)."&language=pt-BR&include_adult=false";
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0');
    
    $response = curl_exec($ch);
    curl_close($ch);
    
    return json_decode($response, true);
}

// Função para buscar detalhes do TMDB
function getTMDBDetails($id, $type = 'movie') {
    $url = "https://api.themoviedb.org/3/{$type}/{$id}?api_key=".TMDB_API_KEY."&language=pt-BR&append_to_response=credits";
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    
    $response = curl_exec($ch);
    curl_close($ch);
    
    return json_decode($response, true);
}

// Função para buscar jogos da API
function fetchGamesFromAPI() {
    $api_url = 'https://meusjogos.shop/dados_api.json';
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $api_url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0');
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($httpCode == 200 && !empty($response)) {
        return json_decode($response, true);
    }
    
    return [];
}

// Função para baixar imagem do jogo
function downloadGameImage($url, $filename) {
    if (empty($url)) return false;
    
    // Limpar URL (remover parâmetros desnecessários)
    if (strpos($url, 'img/api.php') !== false) {
        $parts = parse_url($url);
        parse_str($parts['query'] ?? '', $query);
        if (isset($query['url'])) {
            $url = $query['url'];
        }
    }
    
    $destination = GAMES_IMAGES_DIR . $filename;
    
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    
    $imageData = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($httpCode == 200 && !empty($imageData)) {
        file_put_contents($destination, $imageData);
        return $filename;
    }
    
    return false;
}

// Função para gerar banner
function generateBanner($data) {
    $width = 1200;
    $height = 630;
    
    // Criar imagem
    $banner = imagecreatetruecolor($width, $height);
    
    // Temas disponíveis
    $themes = readJSON(THEMES_FILE);
    $theme = isset($themes[$data['theme']]) ? $themes[$data['theme']] : (isset($themes['default']) ? $themes['default'] : ['bg_color' => '#2c3e50', 'title_color' => '#ffffff']);
    
    // Fundo
    $bg_color = imagecolorallocate($banner, 
        hexdec(substr($theme['bg_color'], 1, 2)), 
        hexdec(substr($theme['bg_color'], 3, 2)), 
        hexdec(substr($theme['bg_color'], 5, 2))
    );
    imagefill($banner, 0, 0, $bg_color);
    
    // Gradiente
    for ($i = 0; $i < $height; $i++) {
        $color = imagecolorallocatealpha($banner, 0, 0, 0, 60 - ($i * 60 / $height));
        imageline($banner, 0, $i, $width, $i, $color);
    }
    
    // Adicionar logo se existir
    if (!empty($data['logo']) && file_exists(LOGOS_DIR . $data['logo'])) {
        $logo_path = LOGOS_DIR . $data['logo'];
        $ext = strtolower(pathinfo($logo_path, PATHINFO_EXTENSION));
        $logo = null;
        if ($ext == 'png') $logo = @imagecreatefrompng($logo_path);
        elseif ($ext == 'jpg' || $ext == 'jpeg') $logo = @imagecreatefromjpeg($logo_path);
        elseif ($ext == 'gif') $logo = @imagecreatefromgif($logo_path);

        if ($logo) {
            $logo_width = imagesx($logo);
            $logo_height = imagesy($logo);
            
            // Redimensionar logo se muito grande
            if ($logo_width > 300) {
                $new_width = 300;
                $new_height = ($logo_height * 300) / $logo_width;
                $resized_logo = imagecreatetruecolor($new_width, $new_height);
                imagealphablending($resized_logo, false);
                imagesavealpha($resized_logo, true);
                imagecopyresampled($resized_logo, $logo, 0, 0, 0, 0, $new_width, $new_height, $logo_width, $logo_height);
                $logo = $resized_logo;
                $logo_width = $new_width;
                $logo_height = $new_height;
            }
            
            imagecopy($banner, $logo, 50, 50, 0, 0, $logo_width, $logo_height);
            imagedestroy($logo);
        }
    }
    
    // Título
    $title_color = imagecolorallocate($banner, 
        hexdec(substr($theme['title_color'], 1, 2)), 
        hexdec(substr($theme['title_color'], 3, 2)), 
        hexdec(substr($theme['title_color'], 5, 2))
    );
    
    // Usar fonte TrueType se disponível, senão usar built-in
    $font_file = FONTS_DIR . 'arial.ttf';
    if (file_exists($font_file)) {
        imagettftext($banner, 48, 0, 50, 400, $title_color, $font_file, $data['title']);
    } else {
        // Fonte built-in
        $font = 5;
        imagestring($banner, $font, 50, 400, $data['title'], $title_color);
    }
    
    // Informações adicionais
    $info_color = imagecolorallocate($banner, 255, 255, 255);
    if (isset($data['year'])) {
        $info_text = $data['year'] . ' • ' . strtoupper($data['type']);
        if (isset($data['rating'])) {
            $info_text .= ' • ' . $data['rating'] . '/10';
        }
        
        if (file_exists($font_file)) {
            imagettftext($banner, 24, 0, 50, 480, $info_color, $font_file, $info_text);
        } else {
            imagestring($banner, 3, 50, 480, $info_text, $info_color);
        }
    }
    
    // Salvar banner
    $filename = uniqid() . '.png';
    $filepath = BANNERS_DIR . $filename;
    imagepng($banner, $filepath);
    imagedestroy($banner);
    
    return $filename;
}

// Função para gerar card VOD
function generateVODCard($data) {
    $width = 500;
    $height = 750;
    
    // Criar imagem
    $card = imagecreatetruecolor($width, $height);
    
    // Carregar temas
    $themes = readJSON(THEMES_VOD_FILE);
    $theme = isset($themes[$data['theme']]) ? $themes[$data['theme']] : (isset($themes['tema1']) ? $themes['tema1'] : ['bg_color' => '#2c3e50', 'title_color' => '#ffffff']);
    
    // Fundo com gradiente
    $bg_color = imagecolorallocate($card, 
        hexdec(substr($theme['bg_color'], 1, 2)), 
        hexdec(substr($theme['bg_color'], 3, 2)), 
        hexdec(substr($theme['bg_color'], 5, 2))
    );
    imagefill($card, 0, 0, $bg_color);
    
    // Gradiente
    for ($i = 0; $i < $height; $i++) {
        $alpha = 80 - ($i * 80 / $height);
        $color = imagecolorallocatealpha($card, 0, 0, 0, $alpha);
        imageline($card, 0, $i, $width, $i, $color);
    }
    
    // Overlay escuro
    $overlay = imagecolorallocatealpha($card, 0, 0, 0, 50);
    imagefilledrectangle($card, 0, 0, $width, $height, $overlay);
    
    // Posições
    $padding = 30;
    $currentY = $padding;
    
    // Logo do site (se existir)
    if (!empty($data['site_logo']) && file_exists(LOGOS_DIR . $data['site_logo'])) {
        $logo_path = LOGOS_DIR . $data['site_logo'];
        $ext = strtolower(pathinfo($logo_path, PATHINFO_EXTENSION));
        $logo = null;
        if ($ext == 'png') $logo = @imagecreatefrompng($logo_path);
        elseif ($ext == 'jpg' || $ext == 'jpeg') $logo = @imagecreatefromjpeg($logo_path);
        elseif ($ext == 'gif') $logo = @imagecreatefromgif($logo_path);

        if ($logo) {
            $logo_width = imagesx($logo);
            $logo_height = imagesy($logo);
            
            // Redimensionar logo
            if ($logo_width > 150) {
                $new_width = 150;
                $new_height = ($logo_height * 150) / $logo_width;
                $resized_logo = imagecreatetruecolor($new_width, $new_height);
                imagealphablending($resized_logo, false);
                imagesavealpha($resized_logo, true);
                imagecopyresampled($resized_logo, $logo, 0, 0, 0, 0, $new_width, $new_height, $logo_width, $logo_height);
                // Liberar a logo original e usar a redimensionada
                imagedestroy($logo);
                $logo = $resized_logo;
                $logo_width = $new_width;
                $logo_height = $new_height;
            }
            
            imagecopy($card, $logo, $padding, $currentY, 0, 0, $logo_width, $logo_height);
            imagedestroy($logo);
            $currentY += $logo_height + 20;
        }
    }
    
    // Título
    $title_color = imagecolorallocate($card, 
        hexdec(substr($theme['title_color'], 1, 2)), 
        hexdec(substr($theme['title_color'], 3, 2)), 
        hexdec(substr($theme['title_color'], 5, 2))
    );
    
    $font_file = FONTS_DIR . 'arial.ttf';
    if (file_exists($font_file)) {
        $title = $data['title'];
        $title = wordwrap($title, 20, "\n");
        $title_lines = explode("\n", $title);
        $title_y = $currentY;
        
        foreach ($title_lines as $line) {
            imagettftext($card, 24, 0, 30, $title_y, $title_color, $font_file, $line);
            $title_y += 35;
        }
        $currentY = $title_y + 10;
        
        // Informações (ano, duração, classificação)
        $info_text = '';
        if (!empty($data['year'])) {
            $info_text .= $data['year'];
        }
        if (!empty($data['runtime'])) {
            $info_text .= ' • ' . $data['runtime'] . 'min';
        }
        if (!empty($data['vote_average'])) {
            $info_text .= ' • ⭐ ' . number_format($data['vote_average'], 1);
        }
        
        if (!empty($info_text)) {
            imagettftext($card, 16, 0, 30, $currentY, $title_color, $font_file, $info_text);
            $currentY += 30;
        }
        
        // Gêneros
        if (!empty($data['genres'])) {
            $genres_text = implode(' • ', array_slice($data['genres'], 0, 3));
            imagettftext($card, 14, 0, 30, $currentY, $title_color, $font_file, $genres_text);
            $currentY += 30;
        }
    }
    
    // Salvar card
    $filename = 'card_' . uniqid() . '.png';
    $filepath = CARDS_DIR . $filename;
    imagepng($card, $filepath);
    imagedestroy($card);
    
    return $filename;
}

// Função para gerar card de jogo
function generateGameCard($game_data, $theme_key) {
    $width = 600;
    $height = 400;
    
    // Criar imagem
    $card = imagecreatetruecolor($width, $height);
    
    // Carregar temas de jogos
    $themes = readJSON(GAMES_THEMES_FILE);
    $theme = isset($themes[$theme_key]) ? $themes[$theme_key] : (isset($themes['tema1']) ? $themes['tema1'] : ['bg_color' => '#1a1a2e']);
    
    // Fundo
    $bg_color = imagecolorallocate($card, 
        hexdec(substr($theme['bg_color'], 1, 2)), 
        hexdec(substr($theme['bg_color'], 3, 2)), 
        hexdec(substr($theme['bg_color'], 5, 2))
    );
    imagefill($card, 0, 0, $bg_color);
    
    // Padrão de fundo (listras diagonais)
    $line_color = imagecolorallocatealpha($card, 255, 255, 255, 90);
    for ($i = -$height; $i < $width; $i += 30) {
        imageline($card, $i, 0, $i + $height, $height, $line_color);
    }
    
    // Logo do site no topo
    $settings = readJSON(SETTINGS_FILE);
    if (!empty($settings['logo']) && file_exists(LOGOS_DIR . $settings['logo'])) {
        $logo_path = LOGOS_DIR . $settings['logo'];
        $ext = strtolower(pathinfo($logo_path, PATHINFO_EXTENSION));
        $logo = null;
        if ($ext == 'png') $logo = @imagecreatefrompng($logo_path);
        elseif ($ext == 'jpg' || $ext == 'jpeg') $logo = @imagecreatefromjpeg($logo_path);
        elseif ($ext == 'gif') $logo = @imagecreatefromgif($logo_path);

        if ($logo) {
            $logo_width = imagesx($logo);
            $logo_height = imagesy($logo);
            
            if ($logo_width > 100) {
                $new_width = 100;
                $new_height = ($logo_height * 100) / $logo_width;
                $resized_logo = imagecreatetruecolor($new_width, $new_height);
                imagealphablending($resized_logo, false);
                imagesavealpha($resized_logo, true);
                imagecopyresampled($resized_logo, $logo, 0, 0, 0, 0, $new_width, $new_height, $logo_width, $logo_height);
                $logo = $resized_logo;
                $logo_width = $new_width;
                $logo_height = $new_height;
            }
            
            imagecopy($card, $logo, 20, 20, 0, 0, $logo_width, $logo_height);
            imagedestroy($logo);
        }
    }
    
    // Status do jogo (AO VIVO, ADIADO, etc)
    if (!empty($game_data['status'])) {
        $status_color = imagecolorallocate($card, 255, 0, 0);
        $status_text = strtoupper($game_data['status']);
        
        // Fundo vermelho para status
        $status_bg = imagecolorallocate($card, 200, 0, 0);
        imagefilledrectangle($card, $width - 150, 20, $width - 20, 50, $status_bg);
        
        $font_file = FONTS_DIR . 'arialbd.ttf';
        if (file_exists($font_file)) {
            imagettftext($card, 14, 0, $width - 130, 40, $status_color, $font_file, $status_text);
        }
    }
    
    // Competição
    $comp_color = imagecolorallocate($card, 255, 255, 255);
    $font_file = FONTS_DIR . 'arial.ttf';
    if (file_exists($font_file)) {
        imagettftext($card, 16, 0, 20, 100, $comp_color, $font_file, $game_data['competicao']);
    }
    
    // Horário
    $time_color = imagecolorallocate($card, 255, 255, 0);
    $time_text = $game_data['horario'] . ' - ' . ucfirst($game_data['data_jogo']);
    if (file_exists($font_file)) {
        imagettftext($card, 14, 0, 20, 130, $time_color, $font_file, $time_text);
    }
    
    // Times
    $centerX = $width / 2;
    $teamY = 200;
    
    // Time 1 (esquerda)
    if (!empty($game_data['img_time1']) && file_exists(GAMES_IMAGES_DIR . $game_data['img_time1'])) {
        $img_path = GAMES_IMAGES_DIR . $game_data['img_time1'];
        $ext = strtolower(pathinfo($img_path, PATHINFO_EXTENSION));
        $team1_img = null;
        if ($ext == 'png') $team1_img = @imagecreatefrompng($img_path);
        elseif ($ext == 'jpg' || $ext == 'jpeg') $team1_img = @imagecreatefromjpeg($img_path);
        elseif ($ext == 'gif') $team1_img = @imagecreatefromgif($img_path);

        if ($team1_img) {
            $img_width = 60;
            $img_height = 60;
            $team1_img = imagescale($team1_img, $img_width, $img_height);
            imagecopy($card, $team1_img, 150 - 30, $teamY - 30, 0, 0, $img_width, $img_height);
            imagedestroy($team1_img);
        }
    }
    
    // Nome Time 1
    $team1_name = wordwrap($game_data['time1'], 15, "\n");
    $team1_lines = explode("\n", $team1_name);
    $team1_y = $teamY + 40;
    foreach ($team1_lines as $line) {
        if (file_exists($font_file)) {
            imagettftext($card, 12, 0, 150 - 40, $team1_y, $comp_color, $font_file, $line);
        }
        $team1_y += 20;
    }
    
    // Placar / VS
    if (!empty($game_data['placar_time1']) && !empty($game_data['placar_time2'])) {
        $placar_text = $game_data['placar_time1'] . ' x ' . $game_data['placar_time2'];
        if (file_exists($font_file)) {
            imagettftext($card, 24, 0, $centerX - 40, $teamY + 10, $time_color, $font_file, $placar_text);
        }
    } else {
        // VS
        $vs_color = imagecolorallocate($card, 255, 255, 0);
        if (file_exists($font_file)) {
            imagettftext($card, 20, 0, $centerX - 20, $teamY + 10, $vs_color, $font_file, 'VS');
        }
    }
    
    // Time 2 (direita)
    if (!empty($game_data['img_time2']) && file_exists(GAMES_IMAGES_DIR . $game_data['img_time2'])) {
        $img_path = GAMES_IMAGES_DIR . $game_data['img_time2'];
        $ext = strtolower(pathinfo($img_path, PATHINFO_EXTENSION));
        $team2_img = null;
        if ($ext == 'png') $team2_img = @imagecreatefrompng($img_path);
        elseif ($ext == 'jpg' || $ext == 'jpeg') $team2_img = @imagecreatefromjpeg($img_path);
        elseif ($ext == 'gif') $team2_img = @imagecreatefromgif($img_path);

        if ($team2_img) {
            $img_width = 60;
            $img_height = 60;
            $team2_img = imagescale($team2_img, $img_width, $img_height);
            imagecopy($card, $team2_img, $width - 150 - 30, $teamY - 30, 0, 0, $img_width, $img_height);
            imagedestroy($team2_img);
        }
    }
    
    // Nome Time 2
    $team2_name = wordwrap($game_data['time2'], 15, "\n");
    $team2_lines = explode("\n", $team2_name);
    $team2_y = $teamY + 40;
    foreach ($team2_lines as $line) {
        if (file_exists($font_file)) {
            imagettftext($card, 12, 0, $width - 150 - 20, $team2_y, $comp_color, $font_file, $line);
        }
        $team2_y += 20;
    }
    
    // Canais
    if (!empty($game_data['canais'])) {
        $canal_y = 300;
        $canal_color = imagecolorallocate($card, 200, 200, 200);
        
        if (file_exists($font_file)) {
            imagettftext($card, 12, 0, 20, $canal_y, $canal_color, $font_file, 'Onde assistir:');
        }
        
        $canal_x = 150;
        foreach ($game_data['canais'] as $index => $canal) {
            if ($index < 3) { // Máximo 3 canais
                if (!empty($canal['img']) && file_exists(GAMES_IMAGES_DIR . $canal['img'])) {
                    $img_path = GAMES_IMAGES_DIR . $canal['img'];
                    $ext = strtolower(pathinfo($img_path, PATHINFO_EXTENSION));
                    $canal_img = null;
                    if ($ext == 'png') $canal_img = @imagecreatefrompng($img_path);
                    elseif ($ext == 'jpg' || $ext == 'jpeg') $canal_img = @imagecreatefromjpeg($img_path);
                    elseif ($ext == 'gif') $canal_img = @imagecreatefromgif($img_path);

                    if ($canal_img) {
                        $canal_img = imagescale($canal_img, 80, 30);
                        imagecopy($card, $canal_img, $canal_x, $canal_y - 25, 0, 0, 80, 30);
                        imagedestroy($canal_img);
                        $canal_x += 90;
                    } else {
                        if (file_exists($font_file)) {
                            imagettftext($card, 10, 0, $canal_x, $canal_y, $canal_color, $font_file, $canal['nome']);
                        }
                        $canal_x += 100;
                    }
                } else {
                    if (file_exists($font_file)) {
                        imagettftext($card, 10, 0, $canal_x, $canal_y, $canal_color, $font_file, $canal['nome']);
                    }
                    $canal_x += 100;
                }
            }
        }
    }
    
    // Salvar card
    $filename = 'game_' . uniqid() . '.png';
    $filepath = GAMES_CARDS_DIR . $filename;
    imagepng($card, $filepath);
    imagedestroy($card);
    
    return $filename;
}

// Temas padrão
function initThemes() {
    $themes = [
        'default' => [
            'name' => 'Azul Metálico',
            'bg_color' => '#2c3e50',
            'title_color' => '#ffffff',
            'accent_color' => '#00ffff'
        ],
        'neon' => [
            'name' => 'Azul Neon',
            'bg_color' => '#000033',
            'title_color' => '#00ffff',
            'accent_color' => '#ff00ff'
        ],
        'acao' => [
            'name' => 'Ação',
            'bg_color' => '#8b0000',
            'title_color' => '#ffffff',
            'accent_color' => '#ffd700'
        ],
        'comedia' => [
            'name' => 'Comédia',
            'bg_color' => '#ff8c00',
            'title_color' => '#000000',
            'accent_color' => '#ffffff'
        ],
        'terror' => [
            'name' => 'Terror',
            'bg_color' => '#1a1a1a',
            'title_color' => '#ff0000',
            'accent_color' => '#800080'
        ],
        'romance' => [
            'name' => 'Romance',
            'bg_color' => '#ff69b4',
            'title_color' => '#ffffff',
            'accent_color' => '#ff1493'
        ]
    ];
    
    writeJSON(THEMES_FILE, $themes);
    return $themes;
}

// Inicializar temas se arquivo estiver vazio
if (empty(readJSON(THEMES_FILE))) {
    initThemes();
}

// Inicializar temas VOD
function initThemesVOD() {
    $themes = [
        'tema1' => [
            'name' => 'Azul Metálico',
            'bg_color' => '#2c3e50',
            'title_color' => '#ffffff',
            'accent_color' => '#00ffff',
            'description' => 'Tema elegante com azul metálico'
        ],
        'tema2' => [
            'name' => 'Azul Neon',
            'bg_color' => '#000033',
            'title_color' => '#00ffff',
            'accent_color' => '#ff00ff',
            'description' => 'Tema vibrante com neon'
        ],
        'tema3' => [
            'name' => 'Vermelho Ação',
            'bg_color' => '#8b0000',
            'title_color' => '#ffffff',
            'accent_color' => '#ffd700',
            'description' => 'Tema para filmes de ação'
        ],
        'tema4' => [
            'name' => 'Roxo Mistério',
            'bg_color' => '#4b0082',
            'title_color' => '#ffffff',
            'accent_color' => '#ff69b4',
            'description' => 'Tema para suspense/terror'
        ],
        'tema5' => [
            'name' => 'Verde Natureza',
            'bg_color' => '#006400',
            'title_color' => '#ffffff',
            'accent_color' => '#32cd32',
            'description' => 'Tema para documentários'
        ]
    ];
    
    writeJSON(THEMES_VOD_FILE, $themes);
    return $themes;
}

if (empty(readJSON(THEMES_VOD_FILE))) {
    initThemesVOD();
}

// Inicializar temas de jogos
function initGameThemes() {
    $themes = [
        'tema1' => [
            'name' => 'Clássico Escuro',
            'bg_color' => '#1a1a2e',
            'accent_color' => '#16213e',
            'text_color' => '#ffffff',
            'description' => 'Tema escuro elegante'
        ],
        'tema2' => [
            'name' => 'Futebol Gramado',
            'bg_color' => '#0b4619',
            'accent_color' => '#116530',
            'text_color' => '#ffffff',
            'description' => 'Tema verde gramado'
        ],
        'tema3' => [
            'name' => 'Estádio Noturno',
            'bg_color' => '#0c1b33',
            'accent_color' => '#1a2e4f',
            'text_color' => '#ffff00',
            'description' => 'Tema noite de jogo'
        ],
        'tema4' => [
            'name' => 'Campeonato',
            'bg_color' => '#4a1c40',
            'accent_color' => '#6a2c5c',
            'text_color' => '#ffd700',
            'description' => 'Tema campeonato'
        ],
        'tema5' => [
            'name' => 'Clássico Vermelho',
            'bg_color' => '#8b0000',
            'accent_color' => '#a52a2a',
            'text_color' => '#ffffff',
            'description' => 'Tema vermelho vibrante'
        ],
        'tema6' => [
            'name' => 'Azul Campeão',
            'bg_color' => '#003153',
            'accent_color' => '#0047ab',
            'text_color' => '#ffd700',
            'description' => 'Tema azul campeão'
        ],
        'tema7' => [
            'name' => 'Moderno Gradiente',
            'bg_color' => '#2c3e50',
            'accent_color' => '#3498db',
            'text_color' => '#ffffff',
            'description' => 'Tema gradiente metálico'
        ],
        'tema8' => [
            'name' => 'Neon Game',
            'bg_color' => '#0f0f1f',
            'accent_color' => '#00ffff',
            'text_color' => '#ff00ff',
            'description' => 'Tema neon vibrante'
        ],
        'tema9' => [
            'name' => 'Clássico Europeu',
            'bg_color' => '#1b2b3a',
            'accent_color' => '#2c3e50',
            'text_color' => '#ffffff',
            'description' => 'Tema futebol europeu'
        ],
        'tema10' => [
            'name' => 'Brasileirão',
            'bg_color' => '#009c3b',
            'accent_color' => '#ffdf00',
            'text_color' => '#002776',
            'description' => 'Tema verde e amarelo'
        ]
    ];
    
    writeJSON(GAMES_THEMES_FILE, $themes);
    return $themes;
}

if (empty(readJSON(GAMES_THEMES_FILE))) {
    initGameThemes();
}

// Função para gerar card com todos os jogos (Estilo Moderno)
function generateAllGamesCard($games_data, $theme_key) {
    $width = 1080;
    $item_height = 280; // Altura maior para o estilo da imagem
    $spacing = 30;
    $header_height = 120;
    $footer_height = 100;
    
    $num_games = count($games_data);
    $height = $header_height + ($num_games * ($item_height + $spacing)) + $footer_height;
    
    // Criar imagem
    $card = imagecreatetruecolor($width, $height);
    
    // Carregar temas de jogos
    $themes = readJSON(GAMES_THEMES_FILE);
    $theme = isset($themes[$theme_key]) ? $themes[$theme_key] : (isset($themes['tema1']) ? $themes['tema1'] : ['bg_color' => '#1a1a2e']);
    
    // Fundo da imagem principal (gramado ou gradiente)
    $bg_color = imagecolorallocate($card, 20, 20, 40);
    imagefill($card, 0, 0, $bg_color);
    
    // Tentar carregar um fundo se existir, senão usar gradiente
    for ($i = 0; $i < $height; $i++) {
        $r = 20 + ($i * 10 / $height);
        $g = 20 + ($i * 10 / $height);
        $b = 50 + ($i * 20 / $height);
        $color = imagecolorallocate($card, $r, $g, $b);
        imageline($card, 0, $i, $width, $i, $color);
    }

    $white = imagecolorallocate($card, 255, 255, 255);
    $yellow = imagecolorallocate($card, 255, 255, 0);
    $blue_box = imagecolorallocate($card, 0, 0, 150);
    $black_trans = imagecolorallocatealpha($card, 0, 0, 0, 60);
    
    $font_bold = FONTS_DIR . 'arialbd.ttf';
    $font_regular = FONTS_DIR . 'arial.ttf';
    
    // Cabeçalho
    if (file_exists($font_bold)) {
        $title_text = "JOGOS DE HOJE - " . date('d/m/Y');
        $bbox = imagettfbbox(30, 0, $font_bold, $title_text);
        $t_width = $bbox[2] - $bbox[0];
        imagettftext($card, 30, 0, ($width - $t_width) / 2, 70, $white, $font_bold, $title_text);
    }
    
    $currentY = $header_height;
    foreach ($games_data as $game) {
        // Bloco arredondado (simulado com retângulo preenchido)
        $box_padding = 40;
        $box_width = $width - ($box_padding * 2);
        
        // Fundo do card do jogo (Azul escuro arredondado)
        // Usando retângulos para simular bordas arredondadas
        $r = 30; // radius
        imagefilledrectangle($card, $box_padding + $r, $currentY, $box_padding + $box_width - $r, $currentY + $item_height, $blue_box);
        imagefilledrectangle($card, $box_padding, $currentY + $r, $box_padding + $box_width, $currentY + $item_height - $r, $blue_box);
        imagefilledellipse($card, $box_padding + $r, $currentY + $r, $r * 2, $r * 2, $blue_box);
        imagefilledellipse($card, $box_padding + $box_width - $r, $currentY + $r, $r * 2, $r * 2, $blue_box);
        imagefilledellipse($card, $box_padding + $r, $currentY + $item_height - $r, $r * 2, $r * 2, $blue_box);
        imagefilledellipse($card, $box_padding + $box_width - $r, $currentY + $item_height - $r, $r * 2, $r * 2, $blue_box);

        // Faixa Central (Preta/Escura para Horário e Competição)
        imagefilledrectangle($card, $width/2 - 150, $currentY, $width/2 + 150, $currentY + 120, $black_trans);
        
        // Horário
        if (file_exists($font_bold)) {
            $time_text = $game['horario'];
            $bbox = imagettfbbox(24, 0, $font_bold, $time_text);
            $tw = $bbox[2] - $bbox[0];
            imagettftext($card, 24, 0, ($width - $tw) / 2, $currentY + 45, $white, $font_bold, $time_text);
            
            // Competição
            $comp_text = strtoupper($game['competicao']);
            $bbox = imagettfbbox(16, 0, $font_bold, $comp_text);
            $cw = $bbox[2] - $bbox[0];
            imagettftext($card, 16, 0, ($width - $cw) / 2, $currentY + 85, $white, $font_bold, $comp_text);
            
            // VS
            imagettftext($card, 24, 0, ($width - 40) / 2, $currentY + 180, $yellow, $font_bold, "VS");
        }

        // Time 1 (Esquerda)
        if (!empty($game['img_time1']) && file_exists(GAMES_IMAGES_DIR . $game['img_time1'])) {
            $img_path = GAMES_IMAGES_DIR . $game['img_time1'];
            $ext = strtolower(pathinfo($img_path, PATHINFO_EXTENSION));
            $img = null;
            if ($ext == 'png') $img = @imagecreatefrompng($img_path);
            elseif ($ext == 'jpg' || $ext == 'jpeg') $img = @imagecreatefromjpeg($img_path);
            elseif ($ext == 'gif') $img = @imagecreatefromgif($img_path);

            if ($img) {
                $img = imagescale($img, 120, 120);
                imagecopy($card, $img, $box_padding + 100, $currentY + 10, 0, 0, 120, 120);
                imagedestroy($img);
            }
        }
        if (file_exists($font_bold)) {
            $name1 = strtoupper($game['time1']);
            $bbox = imagettfbbox(22, 0, $font_bold, $name1);
            $nw = $bbox[2] - $bbox[0];
            imagettftext($card, 22, 0, $box_padding + 160 - ($nw/2), $currentY + 180, $white, $font_bold, $name1);
        }

        // Time 2 (Direita)
        if (!empty($game['img_time2']) && file_exists(GAMES_IMAGES_DIR . $game['img_time2'])) {
            $img_path = GAMES_IMAGES_DIR . $game['img_time2'];
            $ext = strtolower(pathinfo($img_path, PATHINFO_EXTENSION));
            $img = null;
            if ($ext == 'png') $img = @imagecreatefrompng($img_path);
            elseif ($ext == 'jpg' || $ext == 'jpeg') $img = @imagecreatefromjpeg($img_path);
            elseif ($ext == 'gif') $img = @imagecreatefromgif($img_path);

            if ($img) {
                $img = imagescale($img, 120, 120);
                imagecopy($card, $img, $width - $box_padding - 220, $currentY + 10, 0, 0, 120, 120);
                imagedestroy($img);
            }
        }
        if (file_exists($font_bold)) {
            $name2 = strtoupper($game['time2']);
            $bbox = imagettfbbox(22, 0, $font_bold, $name2);
            $nw = $bbox[2] - $bbox[0];
            imagettftext($card, 22, 0, $width - $box_padding - 160 - ($nw/2), $currentY + 180, $white, $font_bold, $name2);
        }

        // Canais (Abaixo do VS)
        if (!empty($game['canais'])) {
            $canais_text = "";
            foreach($game['canais'] as $c) $canais_text .= strtoupper($c['nome'] ?? $c) . " ";
            $canais_text = trim($canais_text);
            if (file_exists($font_bold)) {
                $bbox = imagettfbbox(18, 0, $font_bold, $canais_text);
                $cw = $bbox[2] - $bbox[0];
                imagettftext($card, 18, 0, ($width - $cw) / 2, $currentY + 240, $white, $font_bold, $canais_text);
            }
        }

        $currentY += $item_height + $spacing;
    }
    
    // Salvar card
    $filename = 'all_games_' . uniqid() . '.png';
    $filepath = GAMES_CARDS_DIR . $filename;
    imagepng($card, $filepath);
    imagedestroy($card);
    
    return $filename;
}
?>



?>
